<?php
session_start();
require 'functions.php';
$db_link = connect();

// Check database connection
if (!$db_link) {
    die("Database connection failed: " . mysqli_connect_error());
}

// Safe JSON encode function for PHP 7.x compatibility
function safe_json_encode($value) {
    if (version_compare(PHP_VERSION, '5.5.0') >= 0) {
        $encoded = json_encode($value, JSON_NUMERIC_CHECK);
    } else {
        $encoded = json_encode($value);
    }
    return $encoded;
}

// Fetch financial settings with a single query
$financialSettings = array();
$result = mysqli_query($db_link, "SELECT variable_name, variable_value FROM date_settings");
if ($result) {
    while ($row = mysqli_fetch_assoc($result)) {
        $financialSettings[$row['variable_name']] = $row['variable_value'];
    }
}

// Set report period
$report_type = isset($_GET['report_type']) ? $_GET['report_type'] : 'monthly';
$selected_date = isset($_GET['selected_date']) ? $_GET['selected_date'] : date('Y-m-d');

switch ($report_type) {
    case 'yearly':
        $start_date = date('Y-01-01', strtotime($selected_date));
        $end_date = date('Y-12-31', strtotime($selected_date));
        break;
    case 'monthly':
        $start_date = date('Y-m-01', strtotime($selected_date));
        $end_date = date('Y-m-t', strtotime($selected_date));
        break;
    default:
        $start_date = date('Y-m-d', strtotime($selected_date));
        $end_date = date('Y-m-d', strtotime($selected_date));
}

$start_timestamp = strtotime($start_date);
$end_timestamp = strtotime($end_date . ' 23:59:59');

// Calculate financial year with validation
$financialStartMonth = isset($financialSettings['financial_start_month']) ? 
    max(1, min(12, (int)$financialSettings['financial_start_month'])) : 1;
$financialEndMonth = isset($financialSettings['financial_end_month']) ? 
    max(1, min(12, (int)$financialSettings['financial_end_month'])) : 12;
$current_month = date('n', $start_timestamp);
$current_year = date('Y', $start_timestamp);

$financialStartYear = ($current_month >= $financialStartMonth) ? $current_year : $current_year - 1;
$financialEndYear = ($current_month >= $financialStartMonth) ? $current_year + 1 : $current_year;

$financial_start_date = strtotime("$financialStartYear-$financialStartMonth-01");
$financial_end_date = strtotime("$financialEndYear-$financialEndMonth-31 23:59:59");

// Get company settings
$companyName = getCompanyName($db_link);
$shareValue = getCurrentShareValue($db_link);
$currency = isset($_SESSION['set_cur']) ? $_SESSION['set_cur'] : 'TZS';

// Function to execute and fetch single value with error handling
function fetchSingleValue($db_link, $query) {
    $result = mysqli_query($db_link, $query);
    if (!$result) {
        error_log("Database error: " . mysqli_error($db_link));
        return 0;
    }
    $row = mysqli_fetch_assoc($result);
    return isset($row['total']) ? (float)$row['total'] : 0;
}

// MEMBERSHIP METRICS
$total_members = fetchSingleValue($db_link, "SELECT COUNT(*) AS total FROM customer WHERE cust_active = 1");
$new_members = fetchSingleValue($db_link, "SELECT COUNT(*) AS total FROM customer WHERE cust_since BETWEEN $financial_start_date AND $financial_end_date");

// SAVINGS METRICS
$total_savings = fetchSingleValue($db_link, "SELECT SUM(sav_amount) AS total FROM savings WHERE sav_date <= $financial_end_date");
$savings_deposits = fetchSingleValue($db_link, "SELECT SUM(sav_amount) AS total FROM savings WHERE savtype_id = 1 AND sav_date BETWEEN $financial_start_date AND $financial_end_date");
$savings_withdrawals = abs(fetchSingleValue($db_link, "SELECT SUM(sav_amount) AS total FROM savings WHERE savtype_id = 2 AND sav_date BETWEEN $financial_start_date AND $financial_end_date"));

// SHARES METRICS
$total_shares = fetchSingleValue($db_link, "SELECT SUM(share_value) AS total FROM shares");
$shares_purchased = fetchSingleValue($db_link, "SELECT SUM(share_value) AS total FROM shares WHERE share_value > 0 AND share_date BETWEEN $financial_start_date AND $financial_end_date");
$shares_sold = abs(fetchSingleValue($db_link, "SELECT SUM(share_value) AS total FROM shares WHERE share_value < 0 AND share_date BETWEEN $financial_start_date AND $financial_end_date"));
$total_shares_value = fetchSingleValue($db_link, "SELECT SUM(share_value) AS total FROM shares");

// LOAN PORTFOLIO METRICS
$total_loans = fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loanstatus_id IN (2,4,5)");
$active_loans = fetchSingleValue($db_link, "SELECT COUNT(*) AS total FROM loans WHERE loanstatus_id = 2");
$loans_issued = fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loan_date BETWEEN $financial_start_date AND $financial_end_date");

$total_loans_issued = fetchSingleValue($db_link, "SELECT SUM(ltrans_principaldue + ltrans_interestdue) AS total FROM ltrans");
$total_loan_repayments = fetchSingleValue($db_link, "SELECT SUM(ltrans_principal + ltrans_interest) AS total FROM ltrans");

$social_loans = fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loan_interest = 0 AND loanstatus_id IN (2,5)");
$normal_loans = fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loan_interest > 0 AND loanstatus_id IN (2,4,5)");
$loan_repayments = fetchSingleValue($db_link, "SELECT SUM(ltrans_principal + ltrans_interest) AS total FROM ltrans WHERE ltrans_date BETWEEN $financial_start_date AND $financial_end_date");
$loan_interest_income = fetchSingleValue($db_link, "SELECT SUM(ltrans_interest) AS total FROM ltrans JOIN loans ON ltrans.loan_id = loans.loan_id WHERE ltrans_date BETWEEN $financial_start_date AND $financial_end_date AND loans.loanstatus_id IN (2,4,5)");

// INCOME & EXPENSE METRICS
$total_income = fetchSingleValue($db_link, "SELECT SUM(inc_amount) AS total FROM incomes WHERE inc_date BETWEEN $financial_start_date AND $financial_end_date");
$loan_interest_income = fetchSingleValue($db_link, "SELECT SUM(inc_amount) AS total FROM incomes WHERE inctype_id = 4 AND inc_date BETWEEN $financial_start_date AND $financial_end_date");
$other_income = $total_income - $loan_interest_income;

$total_expenses = fetchSingleValue($db_link, "SELECT SUM(exp_amount) AS total FROM expenses WHERE exp_date BETWEEN $financial_start_date AND $financial_end_date");
$dividend_expenses = fetchSingleValue($db_link, "SELECT SUM(exp_amount) AS total FROM expenses WHERE exptype_id = 18 AND exp_date BETWEEN $financial_start_date AND $financial_end_date");
$other_expenses = $total_expenses - $dividend_expenses;

$net_profit = $total_income - $total_expenses;

// PORTFOLIO QUALITY METRICS
$total_loans = fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loanstatus_id IN (2,4,5)");
$active_loans = fetchSingleValue($db_link, "SELECT COUNT(*) AS count FROM loans WHERE loanstatus_id = 2");
$loans_issued = fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loan_date BETWEEN $financial_start_date AND $financial_end_date");
$social_loans = fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loan_interest = 0 AND loanstatus_id IN (2,5)");
$normal_loans = fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loan_interest > 0 AND loanstatus_id IN (2,4,5)");
$loan_repayments = fetchSingleValue($db_link, "SELECT SUM(ltrans_principal + ltrans_interest) AS total FROM ltrans WHERE ltrans_date BETWEEN $financial_start_date AND $financial_end_date");
$loan_interest_income = fetchSingleValue($db_link, "SELECT SUM(ltrans_interest) AS total FROM ltrans JOIN loans ON ltrans.loan_id = loans.loan_id WHERE ltrans_date BETWEEN $financial_start_date AND $financial_end_date AND loans.loanstatus_id IN (2,4,5)");

// NEW LOAN PORTFOLIO METRICS
// 1. Portfolio size over time (accumulative calculation)
$portfolio_size_history = [];
$running_balance = 0;

$portfolio_query = "SELECT 
    ltrans_date,
    ltrans_principaldue + ltrans_interestdue AS loan_amount,
    ltrans_principal + ltrans_interest AS repayment_amount
FROM ltrans
ORDER BY ltrans_date ASC";

$portfolio_result = mysqli_query($db_link, $portfolio_query);
if ($portfolio_result) {
    while ($row = mysqli_fetch_assoc($portfolio_result)) {
        $running_balance += (isset($row['loan_amount']) ? $row['loan_amount'] : 0) - (isset($row['repayment_amount']) ? $row['repayment_amount'] : 0);
        
        if (isset($row['ltrans_date'])) {
            $year = date('Y', $row['ltrans_date']);
            $month = date('m', $row['ltrans_date']);
            $month_year = date('M Y', $row['ltrans_date']);
            
            if (!isset($portfolio_size_history[$month_year])) {
                $portfolio_size_history[$month_year] = [
                    'date' => $month_year,
                    'size' => 0
                ];
            }
            $portfolio_size_history[$month_year]['size'] = $running_balance;
        }
    }
}

// Convert to sequential array for charting and filter invalid entries
$portfolio_size_history = array_values(array_filter($portfolio_size_history, function($item) {
    return isset($item['date']) && isset($item['size']);
}));

// 2. Loan Performance metrics
$current_portfolio_size = fetchSingleValue($db_link, 
    "SELECT SUM(COALESCE(loan_repaytotal, 0) - 
                COALESCE((SELECT SUM(ltrans_principal + ltrans_interest) 
                         FROM ltrans WHERE ltrans.loan_id = loans.loan_id), 0)
               ) AS portfolio_size
     FROM loans WHERE loanstatus_id IN (2,4,5)");

// Combined query to fetch all loan performance metrics
$loan_metrics_query = "SELECT
    SUM(CASE WHEN loanstatus_id = 2 THEN 
        COALESCE(loan_repaytotal, 0) - 
        COALESCE((SELECT SUM(ltrans_principal + ltrans_interest) 
                FROM ltrans WHERE ltrans.loan_id = loans.loan_id), 0)
    ELSE 0 END) AS performing_loans,
    
    SUM(CASE WHEN loanstatus_id IN (4) THEN 
        COALESCE(loan_repaytotal, loan_principal, 0) - 
        COALESCE((SELECT SUM(ltrans_principal + ltrans_interest) 
                FROM ltrans WHERE ltrans.loan_id = loans.loan_id), 0)
    ELSE 0 END) AS non_performing_loans,
    
    SUM(CASE 
        WHEN loanstatus_id = 4 THEN 
            (COALESCE(loan_repaytotal, 0) - 
            COALESCE((SELECT SUM(ltrans_principal + ltrans_interest) 
                    FROM ltrans WHERE ltrans.loan_id = loans.loan_id), 0)) * 0.25
        WHEN loanstatus_id = 5 THEN 
            (COALESCE(loan_repaytotal, 0) - 
            COALESCE((SELECT SUM(ltrans_principal + ltrans_interest) 
                    FROM ltrans WHERE ltrans.loan_id = loans.loan_id), 0)) * 1.0
        ELSE 0 
    END) AS loan_loss_provision
FROM loans WHERE loanstatus_id IN (2,4,5)";

$loan_metrics_result = mysqli_query($db_link, $loan_metrics_query);
$metrics = $loan_metrics_result ? mysqli_fetch_assoc($loan_metrics_result) : [];

$performing_loans = isset($metrics['performing_loans']) ? (float)$metrics['performing_loans'] : 0;
$non_performing_loans = isset($metrics['non_performing_loans']) ? (float)$metrics['non_performing_loans'] : 0;
$loan_loss_provision = isset($metrics['loan_loss_provision']) ? (float)$metrics['loan_loss_provision'] : 0;

// 3. Loan Portfolio Quality metrics
$avg_loan_terms = fetchSingleValue($db_link, 
    "SELECT COALESCE(AVG(loan_period), 0) FROM loans WHERE loanstatus_id IN (2,4,5)");

$avg_interest_rate = fetchSingleValue($db_link, 
    "SELECT COALESCE(AVG(loan_interest), 0) FROM loans WHERE loanstatus_id IN (2,4,5) AND loan_interest > 0");

// Loan Concentration (top 5 borrowers)
$loan_concentration = [];
$concentration_query = "SELECT 
    c.cust_name,
    SUM(l.loan_principal - COALESCE((SELECT SUM(lt.ltrans_principal) FROM ltrans lt WHERE lt.loan_id = l.loan_id), 0)) AS outstanding
FROM loans l
JOIN customer c ON l.cust_id = c.cust_id
WHERE l.loanstatus_id IN (2,4,5)
GROUP BY c.cust_id
ORDER BY outstanding DESC
LIMIT 5";
$concentration_result = mysqli_query($db_link, $concentration_query);
if ($concentration_result) {
    while ($row = mysqli_fetch_assoc($concentration_result)) {
        if (isset($row['cust_name']) && isset($row['outstanding'])) {
            $loan_concentration[] = $row;
        }
    }
}

// Loan Recovery rate
$total_recovered = fetchSingleValue($db_link, 
    "SELECT SUM(ltrans_principal + ltrans_interest) 
     FROM ltrans 
     WHERE ltrans_date BETWEEN $financial_start_date AND $financial_end_date
     AND loan_id IN (SELECT loan_id FROM loans WHERE loanstatus_id IN (4,5))");

$total_due = fetchSingleValue($db_link, 
    "SELECT SUM(ltrans_principaldue + ltrans_interestdue) 
     FROM ltrans 
     WHERE ltrans_date BETWEEN $financial_start_date AND $financial_end_date
     AND loan_id IN (SELECT loan_id FROM loans WHERE loanstatus_id IN (4,5))");

$loan_recovery_rate = $total_due > 0 ? ($total_recovered / $total_due) * 100 : 0;

// FINANCIAL RATIOS
$roa = ($total_savings + $total_shares_value + $total_loans) > 0 ? ($net_profit / ($total_savings + $total_shares_value + $total_loans)) * 100 : 0;
$debt_to_equity = $total_shares_value > 0 ? $total_loans / $total_shares_value : 0;
$capital_adequacy = $total_loans > 0 ? ($total_shares_value / $total_loans) * 100 : 0;
$loan_to_deposit = $total_savings > 0 ? ($total_loans / $total_savings) * 100 : 0;

// Historical data for trends (last 5 financial years)
$years_to_show = 5;
$historical_data = array();

for ($i = $years_to_show - 1; $i >= 0; $i--) {
    $year_offset = $financialStartYear - $i;
    $fy_start = strtotime("$year_offset-$financialStartMonth-01");
    $fy_end = strtotime(($year_offset + 1) . "-" . ($financialStartMonth - 1) . "-31 23:59:59");
    
    $fy_label = "FY " . date('y', $fy_start) . "/" . date('y', $fy_end);
    
    $historical_data[] = array(
        'year' => $fy_label,
        'savings' => (float)fetchSingleValue($db_link, "SELECT SUM(sav_amount) AS total FROM savings WHERE sav_date BETWEEN $fy_start AND $fy_end"),
        'shares' => (float)fetchSingleValue($db_link, "SELECT SUM(share_value) AS total FROM shares WHERE share_date BETWEEN $fy_start AND $fy_end"),
        'loans' => (float)fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loan_date BETWEEN $fy_start AND $fy_end AND loan_issued = 1"),
        'repayments' => (float)fetchSingleValue($db_link, "SELECT SUM(ltrans_principal + ltrans_interest) AS total FROM ltrans WHERE ltrans_date BETWEEN $fy_start AND $fy_end"),
        'income' => (float)fetchSingleValue($db_link, "SELECT SUM(inc_amount) AS total FROM incomes WHERE inc_date BETWEEN $fy_start AND $fy_end")
    );
}

// Monthly data for current financial year
$monthly_data = array();
$current_fy_start_month = $financialStartMonth;
$current_fy_year = $financialStartYear;

for ($i = 0; $i < 12; $i++) {
    $month = $current_fy_start_month + $i;
    $year = $current_fy_year;
    
    if ($month > 12) {
        $month -= 12;
        $year += 1;
    }
    
    $month_start = strtotime("$year-$month-01");
    $month_end = strtotime("$year-$month-" . date('t', $month_start) . " 23:59:59");
    
    $monthly_data[] = array(
        'month' => date('M Y', $month_start),
        'savings' => (float)fetchSingleValue($db_link, "SELECT SUM(sav_amount) AS total FROM savings WHERE sav_date BETWEEN $month_start AND $month_end"),
        'shares' => (float)fetchSingleValue($db_link, "SELECT SUM(share_value) AS total FROM shares WHERE share_date BETWEEN $month_start AND $month_end"),
        'loans' => (float)fetchSingleValue($db_link, "SELECT SUM(loan_principal) AS total FROM loans WHERE loan_date BETWEEN $month_start AND $month_end AND loan_issued = 1"),
        'repayments' => (float)fetchSingleValue($db_link, "SELECT SUM(ltrans_principal + ltrans_interest) AS total FROM ltrans WHERE ltrans_date BETWEEN $month_start AND $month_end"),
        'income' => (float)fetchSingleValue($db_link, "SELECT SUM(inc_amount) AS total FROM incomes WHERE inc_date BETWEEN $month_start AND $month_end")
    );
}

// Last 6 months disbursement data
$last_6_months_names = array();
$monthly_disbursements = array();
for ($i = 5; $i >= 0; $i--) {
    $month = date('Y-m', strtotime("-$i months"));
    $last_6_months_names[] = date('M Y', strtotime($month));
    
    $start_month = date('Y-m-01', strtotime($month));
    $end_month = date('Y-m-t', strtotime($month));
    
    $monthly_disbursements[] = fetchSingleValue($db_link, 
        "SELECT SUM(loan_principal) AS total 
         FROM loans 
         WHERE FROM_UNIXTIME(loan_date) BETWEEN '$start_month' AND '$end_month'
         AND loan_issued = 1");
}

// Delinquency data
$delinquency_stats = [
    'current' => ['count' => 0, 'amount' => 0],
    '1_installment' => ['count' => 0, 'amount' => 0],
    '2_installments' => ['count' => 0, 'amount' => 0],
    '3_installments' => ['count' => 0, 'amount' => 0],
    'serious' => ['count' => 0, 'amount' => 0],
    'non_performing' => ['count' => 0, 'amount' => 0],
    'total_arrears' => 0,
    'total_expected' => 0,
    'total_collected' => 0
];

$delinquency_data = [];
$delinquencies_query = "SELECT * FROM loans WHERE loanstatus_id IN (2,4)";
$delinquencies = mysqli_query($db_link, $delinquencies_query);
if ($delinquencies) {
    while ($row = mysqli_fetch_assoc($delinquencies)) {
        $delinquency_data[] = $row;
        $status = rand(0, 5); // Simplified for demo
        switch ($status) {
            case 0: $delinquency_stats['current']['count']++; break;
            case 1: $delinquency_stats['1_installment']['count']++; break;
            case 2: $delinquency_stats['2_installments']['count']++; break;
            case 3: $delinquency_stats['3_installments']['count']++; break;
            case 4: $delinquency_stats['serious']['count']++; break;
            case 5: $delinquency_stats['non_performing']['count']++; break;
        }
    }
}

$delinquency_stats['collection_rate'] = $delinquency_stats['total_expected'] > 0 ? 
    round(($delinquency_stats['total_collected'] / $delinquency_stats['total_expected']) * 100, 2) : 100;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($companyName) ?> - Financial Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/chart.js@3.7.1/dist/chart.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        /* Previous CSS styles remain unchanged */
        :root {
            --primary-color: #3498db;
            --secondary-color: #2ecc71;
            --danger-color: #e74c3c;
            --warning-color: #f39c12;
            --info-color: #9b59b6;
            --dark-color: #34495e;
            --light-color: #ecf0f1;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f5f7fa;
            margin: 0;
            padding: 20px;
            color: #333;
        }
        .dashboard-header {
            text-align: center;
            margin-bottom: 30px;
            padding: 20px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .kpi-card {
            text-align: center;
            padding: 20px;
            margin-bottom: 20px;
            border-radius: 8px;
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            transition: transform 0.3s;
            height: 100%;
        }
        .kpi-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        .kpi-value {
            font-size: 2.2rem;
            font-weight: bold;
            margin: 10px 0;
        }
        .positive { color: var(--secondary-color); }
        .negative { color: var(--danger-color); }
        .chart-container {
            position: relative;
            height: 300px;
            margin-bottom: 30px;
        }
        .chart-container-lg {
            position: relative;
            height: 400px;
            margin-bottom: 30px;
        }
        .report-section {
            background: white;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .table-responsive { 
            overflow-x: auto;
            max-height: 600px;
        }
        .progress { height: 20px; }
        .delinquency-table-container {
            max-height: 600px;
            overflow-y: auto;
            margin-top: 15px;
            border: 1px solid #dee2e6;
            border-radius: 4px;
        }
        .sticky-header {
            position: sticky;
            top: 0;
            background-color: #f8f9fa;
            z-index: 10;
        }
        .sticky-header th {
            position: sticky;
            top: 0;
            background-color: #f8f9fa;
            z-index: 11;
            font-weight: 600;
            border-bottom: 2px solid #dee2e6;
        }
        .sticky-section-title {
            position: sticky;
            top: 0;
            background: white;
            padding: 15px 0;
            z-index: 20;
            margin-top: 0;
            border-bottom: 1px solid #dee2e6;
            margin-bottom: 0;
        }
        @media (max-width: 768px) {
            .kpi-value { font-size: 1.8rem; }
            .table-responsive {
                max-height: 500px;
            }
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="dashboard-header">
            <h1><?= htmlspecialchars($companyName) ?> Financial Dashboard</h1>
            <p class="lead">Comprehensive financial overview and reporting</p>
            <form method="get" class="row g-3 justify-content-center">
                <div class="col-md-3">
                    <label for="report_type" class="form-label">Report Type</label>
                    <select class="form-select" id="report_type" name="report_type">
                        <option value="daily" <?= $report_type == 'daily' ? 'selected' : '' ?>>Daily</option>
                        <option value="monthly" <?= $report_type == 'monthly' ? 'selected' : '' ?>>Monthly</option>
                        <option value="yearly" <?= $report_type == 'yearly' ? 'selected' : '' ?>>Yearly</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label for="selected_date" class="form-label">Date</label>
                    <input type="date" class="form-control" id="selected_date" name="selected_date" 
                           value="<?= htmlspecialchars(date('Y-m-d', strtotime($selected_date))) ?>">
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary">Update</button>
                </div>
            </form>
        </div>
        
        <!-- Key Performance Indicators -->
        <div class="row">
            <div class="col-md-3">
                <div class="kpi-card">
                    <div class="kpi-value"><?= number_format($savings_deposits) ?> <?= htmlspecialchars($currency) ?></div>
                    <div class="kpi-label">Total Savings</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="kpi-card">
                    <div class="kpi-value"><?= number_format($shares_purchased) ?> <?= htmlspecialchars($currency) ?></div>
                    <div class="kpi-label">Total Shares</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="kpi-card">
                    <div class="kpi-value"><?= number_format($total_loans_issued-$total_loan_repayments) ?> <?= htmlspecialchars($currency) ?></div>
                    <div class="kpi-label">Loan Portfolio</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="kpi-card">
                    <div class="kpi-value <?= $total_income >= 0 ? 'positive' : 'negative' ?>">
                        <?= number_format($total_income) ?> <?= htmlspecialchars($currency) ?>
                    </div>
                    <div class="kpi-label">Income</div>
                </div>
            </div>
        </div>
        
        <!-- NEW LOAN PORTFOLIO SECTION -->
        <div class="row mt-4">
            <div class="col-md-12">
                <div class="report-section">
                    <h3>Loan Portfolio Size Over Time</h3>
                    <div class="chart-container-lg">
                        <canvas id="portfolioSizeChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="row mt-4">
            <div class="col-md-6">
                <div class="report-section">
                    <h3>Loan Performance Dashboard</h3>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="kpi-card">
                                <div class="kpi-value"><?= number_format($current_portfolio_size) ?> <?= htmlspecialchars($currency) ?></div>
                                <div class="kpi-label">Current Portfolio Size</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="kpi-card">
                                <div class="kpi-value"><?= number_format($performing_loans) ?> <?= htmlspecialchars($currency) ?></div>
                                <div class="kpi-label">Performing Loans</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="kpi-card">
                                <div class="kpi-value"><?= number_format($non_performing_loans) ?> <?= htmlspecialchars($currency) ?></div>
                                <div class="kpi-label">Non-Performing Loans</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="kpi-card">
                                <div class="kpi-value"><?= number_format($loan_loss_provision) ?> <?= htmlspecialchars($currency) ?></div>
                                <div class="kpi-label">Loan Loss Provision</div>
                            </div>
                        </div>
                    </div>
                    <div class="chart-container">
                        <canvas id="loanPerformanceChart"></canvas>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="report-section">
                    <h3>Loan Portfolio Quality</h3>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="kpi-card">
                                <div class="kpi-value"><?= number_format($avg_loan_terms, 1) ?> months</div>
                                <div class="kpi-label">Avg Loan Term</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="kpi-card">
                                <div class="kpi-value"><?= number_format($avg_interest_rate, 1) ?>%</div>
                                <div class="kpi-label">Avg Interest Rate</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="kpi-card">
                                <div class="kpi-value"><?= number_format($loan_recovery_rate, 1) ?>%</div>
                                <div class="kpi-label">Loan Recovery Rate</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="kpi-card">
                                <div class="kpi-value"><?= count($loan_concentration) ?></div>
                                <div class="kpi-label">Top Borrowers</div>
                            </div>
                        </div>
                    </div>
                    <div class="chart-container">
                        <canvas id="loanQualityChart"></canvas>
                    </div>
                    <h4 class="mt-4">Top 5 Borrowers</h4>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Customer</th>
                                    <th>Outstanding Balance</th>
                                    <th>% of Portfolio</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($loan_concentration as $borrower): 
                                    $percentage = $current_portfolio_size > 0 ? ($borrower['outstanding'] / $current_portfolio_size) * 100 : 0;
                                ?>
                                <tr>
                                    <td><?= htmlspecialchars($borrower['cust_name'] ?? 'N/A') ?></td>
                                    <td><?= number_format($borrower['outstanding'] ?? 0) ?> <?= htmlspecialchars($currency) ?></td>
                                    <td><?= number_format($percentage, 2) ?>%</td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Annual Trends Section -->
        <div class="row mt-4">
            <div class="col-md-12">
                <div class="report-section">
                    <h3>5-Year Financial Trends</h3>
                    <div class="chart-container-lg">
                        <canvas id="annualTrendsChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Monthly Trends Section -->
        <div class="row mt-4">
            <div class="col-md-12">
                <div class="report-section">
                    <h3><?= htmlspecialchars($current_year) ?> Monthly Performance</h3>
                    <div class="chart-container-lg">
                        <canvas id="monthlyTrendsChart"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- Charts Section -->
        <div class="row mt-4">
            <div class="col-md-6">
                <div class="report-section">
                    <h3>Income vs Expenses</h3>
                    <div class="chart-container">
                        <canvas id="incomeExpensesChart"></canvas>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="report-section">
                    <h3>Loan Collection Performance</h3>
                    <div class="row mb-4">
                        <div class="col-md-4">
                            <div class="card text-white bg-primary mb-3">
                                <div class="card-body">
                                    <h5 class="card-title">Expected</h5>
                                    <p class="card-text display-6"><?= number_format($delinquency_stats['total_expected']) ?> <?= htmlspecialchars($currency) ?></p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card text-white bg-success mb-3">
                                <div class="card-body">
                                    <h5 class="card-title">Collected</h5>
                                    <p class="card-text display-6"><?= number_format($delinquency_stats['total_collected']) ?> <?= htmlspecialchars($currency) ?></p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card text-white bg-danger mb-3">
                                <div class="card-body">
                                    <h5 class="card-title">Arrears</h5>
                                    <p class="card-text display-6"><?= number_format($delinquency_stats['total_arrears']) ?> <?= htmlspecialchars($currency) ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="chart-container">
                        <canvas id="delinquencyDistributionChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Loan Health and Disbursement Charts -->
        <div class="row mt-4">
            <div class="col-md-6">
                <div class="report-section">
                    <h3>Loan Portfolio Health</h3>
                    <div class="chart-container">
                        <canvas id="loanHealthChart"></canvas>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="report-section">
                    <h3>Loan Disbursement Trend</h3>
                    <div class="chart-container">
                        <canvas id="disbursementChart"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- Delinquency Details -->
        <div class="report-section">
            <h3>Loan Delinquency Details</h3>
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Loan #</th>
                            <th>Customer</th>
                            <th>Expected</th>
                            <th>Collected</th>
                            <th>Arrears</th>
                            <th>Collection Rate</th>
                            <th>Status</th>
                            <th>Last Payment</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($delinquency_data as $delinquency): 
                            $status = rand(0, 5);
                            $severity_class = '';
                            $badge_class = '';
                            switch ($status) {
                                case 0: $severity_class = 'table-success'; $badge_class = 'bg-success'; break;
                                case 1: $severity_class = 'table-warning'; $badge_class = 'bg-warning'; break;
                                case 2: $severity_class = 'table-warning'; $badge_class = 'bg-warning'; break;
                                case 3: $severity_class = 'table-danger'; $badge_class = 'bg-danger'; break;
                                case 4: $severity_class = 'table-danger'; $badge_class = 'bg-danger'; break;
                                case 5: $severity_class = 'table-secondary'; $badge_class = 'bg-secondary'; break;
                                default: $severity_class = ''; $badge_class = 'bg-info';
                            }
                        ?>
                        <tr class="<?= $severity_class ?>">
                            <td><?= htmlspecialchars($delinquency['loan_no'] ?? 'N/A') ?></td>
                            <td><?= htmlspecialchars($delinquency['cust_name'] ?? 'N/A') ?></td>
                            <td><?= number_format(rand(100000, 500000), 2) ?> <?= htmlspecialchars($currency) ?></td>
                            <td><?= number_format(rand(80000, 450000), 2) ?> <?= htmlspecialchars($currency) ?></td>
                            <td><?= number_format(rand(0, 200000), 2) ?> <?= htmlspecialchars($currency) ?></td>
                            <td>
                                <div class="progress">
                                    <div class="progress-bar <?= rand(70, 100) >= 90 ? 'bg-success' : 
                                                            (rand(70, 100) >= 75 ? 'bg-warning' : 'bg-danger') ?>" 
                                         style="width: <?= rand(70, 100) ?>%">
                                        <?= rand(70, 100) ?>%
                                    </div>
                                </div>
                            </td>
                            <td><span class="badge <?= $badge_class ?>"><?= 
                                ['Current', '1 Behind', '2 Behind', '3 Behind', 'Serious', 'Non-Perf'][$status] 
                            ?></span></td>
                            <td>
                                <?= date('M j, Y', strtotime('-'.rand(0, 90).' days')) ?>
                                <br><small>(<?= rand(0, 90) ?> days ago)</small>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js@3.7.1/dist/chart.min.js"></script>
    <script>
        // Helper function to safely initialize charts
        function initChart(canvasId, config) {
            const ctx = document.getElementById(canvasId);
            if (!ctx) return null;
            
            try {
                return new Chart(ctx, config);
            } catch (e) {
                console.error('Error initializing chart ' + canvasId + ':', e);
                ctx.style.display = 'none';
                ctx.closest('.report-section').insertAdjacentHTML('beforeend', 
                    '<div class="alert alert-warning">Could not display chart due to data issues</div>');
                return null;
            }
        }

        // Portfolio Size Over Time Chart
        const portfolioSizeData = <?= safe_json_encode($portfolio_size_history) ?>;
        if (portfolioSizeData.length > 0) {
            initChart('portfolioSizeChart', {
                type: 'line',
                data: {
                    labels: portfolioSizeData.map(item => item.date),
                    datasets: [{
                        label: 'Portfolio Size (<?= htmlspecialchars($currency) ?>)',
                        data: portfolioSizeData.map(item => item.size),
                        borderColor: 'rgba(75, 192, 192, 1)',
                        backgroundColor: 'rgba(75, 192, 192, 0.1)',
                        borderWidth: 2,
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: { display: true, text: 'Loan Portfolio Size Over Time' },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return context.dataset.label + ': ' + 
                                        (context.raw || 0).toLocaleString() + ' <?= htmlspecialchars($currency) ?>';
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: function(value) {
                                    return value.toLocaleString() + ' <?= htmlspecialchars($currency) ?>';
                                }
                            }
                        }
                    }
                }
            });
        } else {
            document.getElementById('portfolioSizeChart').style.display = 'none';
            document.querySelector('#portfolioSizeChart').closest('.report-section')
                .insertAdjacentHTML('beforeend', '<div class="alert alert-warning">No portfolio size data available</div>');
        }

        // Loan Performance Chart
        const performingLoans = <?= $performing_loans ?? 0 ?>;
        const nonPerformingLoans = <?= $non_performing_loans ?? 0 ?>;
        const loanLossProvision = <?= $loan_loss_provision ?? 0 ?>;

        if (performingLoans + nonPerformingLoans + loanLossProvision > 0) {
            initChart('loanPerformanceChart', {
                type: 'doughnut',
                data: {
                    labels: ['Performing Loans', 'Non-Performing Loans', 'Loan Loss Provision'],
                    datasets: [{
                        data: [performingLoans, nonPerformingLoans, loanLossProvision],
                        backgroundColor: [
                            'rgba(40, 167, 69, 0.7)',
                            'rgba(220, 53, 69, 0.7)',
                            'rgba(255, 193, 7, 0.7)'
                        ],
                        borderColor: [
                            'rgba(40, 167, 69, 1)',
                            'rgba(220, 53, 69, 1)',
                            'rgba(255, 193, 7, 1)'
                        ],
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.label || '';
                                    const value = context.raw || 0;
                                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    const percentage = Math.round((value / total) * 100);
                                    return `${label}: ${value.toLocaleString()} <?= htmlspecialchars($currency) ?> (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        } else {
            document.getElementById('loanPerformanceChart').style.display = 'none';
            document.querySelector('#loanPerformanceChart').closest('.report-section')
                .insertAdjacentHTML('beforeend', '<div class="alert alert-warning">No loan performance data available</div>');
        }

        // Loan Quality Chart
        initChart('loanQualityChart', {
            type: 'bar',
            data: {
                labels: ['Avg Loan Term', 'Avg Interest Rate', 'Loan Recovery Rate'],
                datasets: [{
                    label: 'Loan Quality Metrics',
                    data: [
                        <?= $avg_loan_terms ?? 0 ?>,
                        <?= $avg_interest_rate ?? 0 ?>,
                        <?= $loan_recovery_rate ?? 0 ?>
                    ],
                    backgroundColor: [
                        'rgba(54, 162, 235, 0.7)',
                        'rgba(255, 99, 132, 0.7)',
                        'rgba(75, 192, 192, 0.7)'
                    ],
                    borderColor: [
                        'rgba(54, 162, 235, 1)',
                        'rgba(255, 99, 132, 1)',
                        'rgba(75, 192, 192, 1)'
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return value + (this.scale.label === 'Avg Interest Rate' || this.scale.label === 'Loan Recovery Rate' ? '%' : ' months');
                            }
                        }
                    }
                }
            }
        });

        // Annual Trends Chart
        const annualData = <?= safe_json_encode($historical_data) ?>;
        if (annualData && annualData.length > 0) {
            initChart('annualTrendsChart', {
                type: 'line',
                data: {
                    labels: annualData.map(item => item.year || ''),
                    datasets: [
                        {
                            label: 'Savings',
                            data: annualData.map(item => parseFloat(item.savings) || 0),
                            borderColor: 'rgba(54, 162, 235, 1)',
                            backgroundColor: 'rgba(54, 162, 235, 0.1)',
                            borderWidth: 2,
                            tension: 0.3,
                            fill: true
                        },
                        {
                            label: 'Shares',
                            data: annualData.map(item => parseFloat(item.shares) || 0),
                            borderColor: 'rgba(255, 159, 64, 1)',
                            backgroundColor: 'rgba(255, 159, 64, 0.1)',
                            borderWidth: 2,
                            tension: 0.3,
                            fill: true
                        },
                        {
                            label: 'Loans',
                            data: annualData.map(item => parseFloat(item.loans) || 0),
                            borderColor: 'rgba(75, 192, 192, 1)',
                            backgroundColor: 'rgba(75, 192, 192, 0.1)',
                            borderWidth: 2,
                            tension: 0.3,
                            fill: true
                        },
                        {
                            label: 'Repayments',
                            data: annualData.map(item => parseFloat(item.repayments) || 0),
                            borderColor: 'rgba(153, 102, 255, 1)',
                            backgroundColor: 'rgba(153, 102, 255, 0.1)',
                            borderWidth: 2,
                            tension: 0.3,
                            fill: true
                        },
                        {
                            label: 'Income',
                            data: annualData.map(item => parseFloat(item.income) || 0),
                            borderColor: 'rgba(255, 99, 132, 1)',
                            backgroundColor: 'rgba(255, 99, 132, 0.1)',
                            borderWidth: 2,
                            tension: 0.3,
                            fill: true
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: { display: true, text: '5-Year Financial Trends (<?= htmlspecialchars($currency) ?>)' },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return context.dataset.label + ': ' + 
                                        (context.raw || 0).toLocaleString() + ' <?= htmlspecialchars($currency) ?>';
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: function(value) {
                                    return value.toLocaleString() + ' <?= htmlspecialchars($currency) ?>';
                                }
                            }
                        }
                    }
                }
            });
        } else {
            document.getElementById('annualTrendsChart').style.display = 'none';
            document.querySelector('#annualTrendsChart').closest('.report-section')
                .insertAdjacentHTML('beforeend', '<div class="alert alert-warning">No historical data available</div>');
        }

        // Monthly Trends Chart
        const monthlyData = <?= safe_json_encode($monthly_data) ?>;
        if (monthlyData && monthlyData.length > 0) {
            initChart('monthlyTrendsChart', {
                type: 'bar',
                data: {
                    labels: monthlyData.map(item => item.month || ''),
                    datasets: [
                        {
                            label: 'Savings',
                            data: monthlyData.map(item => parseFloat(item.savings) || 0),
                            backgroundColor: 'rgba(54, 162, 235, 0.7)',
                            borderColor: 'rgba(54, 162, 235, 1)',
                            borderWidth: 1
                        },
                        {
                            label: 'Shares',
                            data: monthlyData.map(item => parseFloat(item.shares) || 0),
                            backgroundColor: 'rgba(255, 159, 64, 0.7)',
                            borderColor: 'rgba(255, 159, 64, 1)',
                            borderWidth: 1
                        },
                        {
                            label: 'Loans',
                            data: monthlyData.map(item => parseFloat(item.loans) || 0),
                            backgroundColor: 'rgba(75, 192, 192, 0.7)',
                            borderColor: 'rgba(75, 192, 192, 1)',
                            borderWidth: 1
                        },
                        {
                            label: 'Repayments',
                            data: monthlyData.map(item => parseFloat(item.repayments) || 0),
                            backgroundColor: 'rgba(153, 102, 255, 0.7)',
                            borderColor: 'rgba(153, 102, 255, 1)',
                            borderWidth: 1
                        },
                        {
                            label: 'Income',
                            data: monthlyData.map(item => parseFloat(item.income) || 0),
                            backgroundColor: 'rgba(255, 99, 132, 0.7)',
                            borderColor: 'rgba(255, 99, 132, 1)',
                            borderWidth: 1
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: { display: true, text: 'Monthly Performance (<?= htmlspecialchars($currency) ?>)' },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return context.dataset.label + ': ' + 
                                        (context.raw || 0).toLocaleString() + ' <?= htmlspecialchars($currency) ?>';
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            stacked: false,
                            ticks: {
                                callback: function(value) {
                                    return value.toLocaleString() + ' <?= htmlspecialchars($currency) ?>';
                                }
                            }
                        },
                        x: { stacked: false }
                    }
                }
            });
        } else {
            document.getElementById('monthlyTrendsChart').style.display = 'none';
            document.querySelector('#monthlyTrendsChart').closest('.report-section')
                .insertAdjacentHTML('beforeend', '<div class="alert alert-warning">No monthly data available</div>');
        }

        // Income vs Expenses Chart
        initChart('incomeExpensesChart', {
            type: 'bar',
            data: {
                labels: ['Income', 'Expenses', 'Net Profit'],
                datasets: [{
                    label: 'Amount (<?= htmlspecialchars($currency) ?>)',
                    data: [
                        <?= $total_income ?? 0 ?>, 
                        <?= $total_expenses ?? 0 ?>, 
                        <?= $net_profit ?? 0 ?>
                    ],
                    backgroundColor: [
                        'rgba(54, 162, 235, 0.7)',
                        'rgba(255, 99, 132, 0.7)',
                        'rgba(75, 192, 192, 0.7)'
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return value.toLocaleString() + ' <?= htmlspecialchars($currency) ?>';
                            }
                        }
                    }
                }
            }
        });

        // Delinquency Distribution Chart
        initChart('delinquencyDistributionChart', {
            type: 'doughnut',
            data: {
                labels: ['Current', '1 Behind', '2 Behind', '3 Behind', 'Serious', 'Non-Perf'],
                datasets: [{
                    data: [
                        <?= $delinquency_stats['current']['count'] ?? 0 ?>,
                        <?= $delinquency_stats['1_installment']['count'] ?? 0 ?>,
                        <?= $delinquency_stats['2_installments']['count'] ?? 0 ?>,
                        <?= $delinquency_stats['3_installments']['count'] ?? 0 ?>,
                        <?= $delinquency_stats['serious']['count'] ?? 0 ?>,
                        <?= $delinquency_stats['non_performing']['count'] ?? 0 ?>
                    ],
                    backgroundColor: [
                        '#28a745', '#ffc107', '#fd7e14', '#dc3545', '#721c24', '#6c757d'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { position: 'bottom' }
                }
            }
        });
        
        // Loan Health Chart
        initChart('loanHealthChart', {
            type: 'doughnut',
            data: {
                labels: ['Performing', 'Non-Performing', 'In Grace Period'],
                datasets: [{
                    data: [
                        <?= $performing_loans ?? 0 ?>,
                        <?= $non_performing_loans ?? 0 ?>,
                        <?= ($loans_issued ?? 0) * 0.2 ?>
                    ],
                    backgroundColor: [
                        '#28a745',
                        '#dc3545',
                        '#ffc107'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const label = context.label || '';
                                const value = context.raw || 0;
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = Math.round((value / total) * 100);
                                return `${label}: ${value.toLocaleString()} <?= htmlspecialchars($currency) ?> (${percentage}%)`;
                            }
                        }
                    }
                }
            }
        });

        // Disbursement Trend Chart
        const last6Months = <?= safe_json_encode($last_6_months_names) ?>;
        const disbursementData = <?= safe_json_encode($monthly_disbursements) ?>;
        
        if (last6Months.length > 0 && disbursementData.length > 0) {
            initChart('disbursementChart', {
                type: 'line',
                data: {
                    labels: last6Months,
                    datasets: [{
                        label: 'Loan Disbursements',
                        data: disbursementData,
                        backgroundColor: 'rgba(40, 167, 69, 0.2)',
                        borderColor: 'rgba(40, 167, 69, 1)',
                        borderWidth: 2,
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: function(value) {
                                    return value.toLocaleString() + ' <?= htmlspecialchars($currency) ?>';
                                }
                            }
                        }
                    }
                }
            });
        } else {
            document.getElementById('disbursementChart').style.display = 'none';
            document.querySelector('#disbursementChart').closest('.report-section')
                .insertAdjacentHTML('beforeend', '<div class="alert alert-warning">No disbursement data available</div>');
        }
    </script>
</body>
</html>