<?php
// Enable error reporting (optional, for debugging)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include the functions file (ensure this path is correct)
require 'functions.php';

// Main webhook logic
$rawData = file_get_contents('php://input');
$smsData = json_decode($rawData, true);

// Check if JSON decoding was successful
if (json_last_error() === JSON_ERROR_NONE) {
    // Extract data from the payload
    $id = $smsData['id']; // Top-level ID from the webhook
    $messageId = $smsData['payload']['messageId']; // Message ID from the payload
    $from = $smsData['payload']['phoneNumber']; // Sender's phone number or ID (e.g., M-PESA)
    $message = $smsData['payload']['message']; // SMS message content
    $timestamp = $smsData['payload']['receivedAt']; // Timestamp of the SMS

    // Log the sender and message for debugging
    error_log("Sender: $from");
    error_log("Message: $message");

    // Normalize the phone number: Remove '+' and ensure it starts with '255'
    $normalizedFrom = preg_replace('/^\+/', '', $from); // Remove '+'
    if (substr($normalizedFrom, 0, 3) !== '255') {
        $normalizedFrom = '255' . substr($normalizedFrom, -9); // Ensure it starts with '255'
    }

    // Log the normalized phone number for debugging
    error_log("Normalized phone number: $normalizedFrom");

    // Connect to the database
    $db_link = connect();
    if ($db_link->connect_error) {
        error_log("Database connection failed: " . $db_link->connect_error);
        die("Database connection failed: " . $db_link->connect_error);
    }

    // Insert the received SMS into the sms_messages table
    $type = 'inbound'; // Assuming the webhook is for inbound messages
    $status = 'received'; // Default status for received messages
    $direction = 'in'; // Direction is 'in' for inbound messages

    $query = 'INSERT INTO sms_messages (id, type, phone_number, message, timestamp, status, direction) VALUES (?, ?, ?, ?, ?, ?, ?)';
    $stmt = $db_link->prepare($query);
    if (!$stmt) {
        error_log("Prepare failed: " . $db_link->error);
        die("Prepare failed: " . $db_link->error);
    }

    // Bind parameters to the statement
    $stmt->bind_param('sssssss', $id, $type, $from, $message, $timestamp, $status, $direction);

    // Execute the statement
    if (!$stmt->execute()) {
        error_log("Execute failed: " . $stmt->error);
        die("Execute failed: " . $stmt->error);
    }

    // Close the statement
    $stmt->close();

    // Log successful insertion
    error_log("Inbound SMS inserted into database successfully.");

    // Convert the message to uppercase for case-insensitive comparison
    $messageUpper = strtoupper($message);

    // Check for different keyword scenarios
    if (strpos($messageUpper, 'REPORT') !== false) {
        processReport($db_link, $normalizedFrom, $message);
    }
    elseif (strpos($messageUpper, 'AMENUNUA HISA') !== false) {
        // Check if the message is from M-KOBA
        if (strtoupper($from) === 'M-KOBA') {
            processAmenunuaHisa($db_link, $normalizedFrom, $message);
        } else {
            error_log("Message is not from M-KOBA. Sender: $from");
        }
    }
    elseif (strpos($messageUpper, 'AMELIPA') !== false && strpos($messageUpper, 'MFUKO WA JAMII') !== false) {
        // Check if the message is from M-KOBA
        if (strtoupper($from) === 'M-KOBA') {
            // Check if the message contains both required keywords
            AmelipaJamii($db_link, $normalizedFrom, $message);
        } else {
            error_log("Message is not from M-KOBA. Sender: $from");
        }
    }
    // Check for loan approval/rejection messages
    elseif (preg_match('/(APPROVED|REJECTED) (\w+)/i', $message, $matches)) {
        $action = strtolower($matches[1]);
        $code = $matches[2];
        $approval = getLoanApprovalByCode($code);
        
        if ($approval && $approval['status'] == 'pending') {
            updateLoanApprovalStatus($approval['id'], $action);
            
          //  if (allLoanApprovalsAreApproved($approval['loan_id'])) {
            //    updateLoanStatus($approval['loan_id'], 'approved');
              //  triggerPayment($approval['loan_id']);
          //  }
            
            // Send confirmation message
            $confirmation = "Your have $action the Loan with code: $code. NUFAIKA ENGIE Sacco" . ($action == 'approve' ? ' ' : ' ');
            sendSMS($normalizedFrom, $confirmation, $db_link);
        } else {
            error_log("Invalid loan approval code or status not pending: $code");
        }
    }
    else {
        error_log("No valid keyword found in message.");
    }
    
    // Close the database connection
    mysqli_close($db_link);

    // Respond with success
    http_response_code(200);
    echo json_encode(['status' => 'success', 'message' => 'Webhook received and data processed']);
} else {
    // Handle JSON decoding error
    error_log("Invalid JSON data: " . json_last_error_msg());
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Invalid JSON data']);
    exit;
}
?>